<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Cotizaciones extends CI_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->library('auth_check');
        $this->auth_check->verificar();
        
        $this->load->model('cotizacion_model');
        $this->load->model('cliente_model');
        $this->load->helper(['url','security']);
        $this->load->library('session');
        $this->load->database();
    }

    /** ===== Helpers internos ===== */

    private function currency_symbol($code) {
        // Si prefieres mostrar "US$" para USD, cámbialo aquí.
        switch (strtoupper((string)$code)) {
            case 'USD': return '$';
            case 'MXN':
            default:    return '$';
        }
    }

    private function currency_suffix($code) {
        // Sufijo para dejar claro la divisa.
        switch (strtoupper((string)$code)) {
            case 'USD': return 'USD';
            case 'MXN':
            default:    return 'MXN';
        }
    }

    private function fmt_money($amount, $code) {
        return $this->currency_symbol($code) . ' ' . number_format((float)$amount, 2, '.', ',') . ' ' . $this->currency_suffix($code);
    }

    /** ===== Vistas ===== */

    public function index() {
        $data['cotizaciones'] = $this->cotizacion_model->get_all();
        $this->load->view('cotizaciones/lista', $data);
    }

    public function crear() {
        $data['clientes'] = $this->cliente_model->get_all();
        $data['numero']   = $this->cotizacion_model->generate_numero();
        // defaults de UI (no obligatorios)
        $data['moneda_default'] = 'MXN';
        $data['iva_default']    = 16;
        $data['aplicar_iva']    = 1;
        $this->load->view('cotizaciones/crear', $data);
    }

    /** ===== Guardar (crear) ===== */
    public function guardar() {
        // ------ Entradas cabecera ------
        $cliente_id     = $this->input->post('cliente_id', TRUE);
        $fecha          = $this->input->post('fecha', TRUE);
        $vigencia_dias  = (int)$this->input->post('vigencia_dias');
        $tipo_inversion = $this->input->post('tipo_inversion', TRUE);
        $observaciones  = $this->input->post('observaciones', TRUE);

        // Moneda / IVA (nuevos)
        $moneda       = $this->input->post('moneda', TRUE);
        $aplicar_iva  = $this->input->post('aplicar_iva') ? 1 : 0;
        $iva_pct_in   = (float)$this->input->post('iva_porcentaje');

        // Normalizar moneda e IVA
        $moneda = in_array($moneda, ['MXN','USD']) ? $moneda : 'MXN';
        if ($moneda === 'USD') {
            // típico exportación 0% salvo que el usuario mande algo distinto
            $iva_porcentaje = $aplicar_iva ? max(0, $iva_pct_in) : 0;
        } else {
            $iva_porcentaje = $aplicar_iva ? ($iva_pct_in > 0 ? $iva_pct_in : 16) : 0;
        }
        $iva_porcentaje = min(max($iva_porcentaje, 0), 100);

        // ------ Ítems ------
        $servicios     = (array)$this->input->post('servicio');
        $descripciones = (array)$this->input->post('descripcion');
        $cantidades    = (array)$this->input->post('cantidad');
        $precios       = (array)$this->input->post('precio_unitario');

        // Descuento (monto en la moneda de la cotización)
        $descuento_str = $this->input->post('descuento');
        $descuento     = (float)str_replace(',', '', (string)$descuento_str);
        if ($descuento < 0) $descuento = 0;

        // ------ Recalcular subtotal desde ítems ------
        $subtotal = 0.0;
        $numRows  = max(count($servicios), count($cantidades), count($precios));
        for ($i = 0; $i < $numRows; $i++) {
            $srv = isset($servicios[$i]) ? trim((string)$servicios[$i]) : '';
            if ($srv === '') continue;

            $qty = isset($cantidades[$i]) ? (float)$cantidades[$i] : 0.0;
            $pu  = isset($precios[$i])    ? (float)$precios[$i]    : 0.0;

            $subtotal += ($qty * $pu);
        }
        $subtotal = round($subtotal, 2);

        if ($descuento > $subtotal) $descuento = $subtotal;
        $base = round($subtotal - $descuento, 2);

        $iva   = $aplicar_iva ? round($base * ($iva_porcentaje / 100), 2) : 0.00;
        $total = round($base + $iva, 2);

        $numero = $this->cotizacion_model->generate_numero();

        // ------ Transacción: cabecera + items ------
        $this->db->trans_start();

        $cotizacion_data = array(
            'numero'         => $numero,
            'cliente_id'     => $cliente_id,
            'fecha'          => $fecha,
            'vigencia_dias'  => $vigencia_dias,
            'tipo_inversion' => $tipo_inversion,

            'moneda'         => $moneda,            // NUEVO
            'aplicar_iva'    => $aplicar_iva,       // NUEVO
            'iva_porcentaje' => $iva_porcentaje,    // NUEVO

            'subtotal'       => $subtotal,
            'descuento'      => $descuento,
            'iva'            => $iva,
            'total'          => $total,

            'observaciones'  => $observaciones
        );

        $cotizacion_id = $this->cotizacion_model->insert($cotizacion_data);

        for ($i = 0; $i < $numRows; $i++) {
            $srv = isset($servicios[$i]) ? trim((string)$servicios[$i]) : '';
            if ($srv === '') continue;

            $desc = isset($descripciones[$i]) ? (string)$descripciones[$i] : '';
            $qty  = isset($cantidades[$i])    ? (float)$cantidades[$i]    : 0.0;
            $pu   = isset($precios[$i])       ? (float)$precios[$i]       : 0.0;

            $item_data = array(
                'cotizacion_id'   => $cotizacion_id,
                'servicio'        => $srv,
                'descripcion'     => $desc,
                'cantidad'        => $qty,
                'precio_unitario' => $pu,
                'total'           => round($qty * $pu, 2),
                'orden'           => $i
                // Si tu tabla de items tiene 'moneda', puedes agregarlo aquí.
                // 'moneda'       => $moneda,
            );
            $this->cotizacion_model->insert_item($item_data);
        }

        $this->db->trans_complete();

        if (!$cotizacion_id || $this->db->trans_status() === FALSE) {
            $this->session->set_flashdata('error', 'No se pudo crear la cotización. Inténtalo de nuevo.');
            redirect('cotizaciones/crear');
            return;
        }

        $this->session->set_flashdata('success', 'Cotización creada exitosamente');
        redirect('cotizaciones/ver/' . $cotizacion_id);
    }

    public function ver($id) {
        $data['cotizacion'] = $this->cotizacion_model->get_by_id($id);
        $data['items']      = $this->cotizacion_model->get_items($id);
        $this->load->view('cotizaciones/ver', $data);
    }

    public function editar($id) {
        $data['cotizacion'] = $this->cotizacion_model->get_by_id($id);
        if (!$data['cotizacion']) {
            $this->session->set_flashdata('error', 'Cotización no encontrada');
            redirect('cotizaciones');
        }
        $data['items']    = $this->cotizacion_model->get_items($id);
        $data['clientes'] = $this->cliente_model->get_all();
        $this->load->view('cotizaciones/editar', $data);
    }

    /** ===== Actualizar ===== */
    public function actualizar($id) {
        // ------ Entradas cabecera ------
        $cliente_id     = $this->input->post('cliente_id', TRUE);
        $fecha          = $this->input->post('fecha', TRUE);
        $vigencia_dias  = (int)$this->input->post('vigencia_dias');
        $tipo_inversion = $this->input->post('tipo_inversion', TRUE);
        $observaciones  = $this->input->post('observaciones', TRUE);

        // Moneda / IVA
        $moneda       = $this->input->post('moneda', TRUE);
        $aplicar_iva  = $this->input->post('aplicar_iva') ? 1 : 0;
        $iva_pct_in   = (float)$this->input->post('iva_porcentaje');
        $moneda       = in_array($moneda, ['MXN','USD']) ? $moneda : 'MXN';
        if ($moneda === 'USD') {
            $iva_porcentaje = $aplicar_iva ? max(0, $iva_pct_in) : 0;
        } else {
            $iva_porcentaje = $aplicar_iva ? ($iva_pct_in > 0 ? $iva_pct_in : 16) : 0;
        }
        $iva_porcentaje = min(max($iva_porcentaje, 0), 100);

        // ------ Ítems ------
        $servicios     = (array)$this->input->post('servicio');
        $descripciones = (array)$this->input->post('descripcion');
        $cantidades    = (array)$this->input->post('cantidad');
        $precios       = (array)$this->input->post('precio_unitario');

        $descuento_str = $this->input->post('descuento');
        $descuento     = (float)str_replace(',', '', (string)$descuento_str);
        if ($descuento < 0) $descuento = 0;

        // ------ Recalcular subtotal desde ítems ------
        $subtotal = 0.0;
        $numRows  = max(count($servicios), count($cantidades), count($precios));
        for ($i = 0; $i < $numRows; $i++) {
            $srv = isset($servicios[$i]) ? trim((string)$servicios[$i]) : '';
            if ($srv === '') continue;

            $qty = isset($cantidades[$i]) ? (float)$cantidades[$i] : 0.0;
            $pu  = isset($precios[$i])    ? (float)$precios[$i]    : 0.0;

            $subtotal += ($qty * $pu);
        }
        $subtotal = round($subtotal, 2);

        if ($descuento > $subtotal) $descuento = $subtotal;
        $base  = round($subtotal - $descuento, 2);
        $iva   = $aplicar_iva ? round($base * ($iva_porcentaje / 100), 2) : 0.00;
        $total = round($base + $iva, 2);

        // ------ Transacción ------
        $this->db->trans_start();

        $cotizacion_data = array(
            'cliente_id'     => $cliente_id,
            'fecha'          => $fecha,
            'vigencia_dias'  => $vigencia_dias,
            'tipo_inversion' => $tipo_inversion,

            'moneda'         => $moneda,            // NUEVO
            'aplicar_iva'    => $aplicar_iva,       // NUEVO
            'iva_porcentaje' => $iva_porcentaje,    // NUEVO

            'subtotal'       => $subtotal,
            'descuento'      => $descuento,
            'iva'            => $iva,
            'total'          => $total,

            'observaciones'  => $observaciones
        );
        $this->cotizacion_model->update($id, $cotizacion_data);

        $this->cotizacion_model->delete_items($id);
        for ($i = 0; $i < $numRows; $i++) {
            $srv = isset($servicios[$i]) ? trim((string)$servicios[$i]) : '';
            if ($srv === '') continue;

            $desc = isset($descripciones[$i]) ? (string)$descripciones[$i] : '';
            $qty  = isset($cantidades[$i])    ? (float)$cantidades[$i]    : 0.0;
            $pu   = isset($precios[$i])       ? (float)$precios[$i]       : 0.0;

            $item_data = array(
                'cotizacion_id'   => $id,
                'servicio'        => $srv,
                'descripcion'     => $desc,
                'cantidad'        => $qty,
                'precio_unitario' => $pu,
                'total'           => round($qty * $pu, 2),
                'orden'           => $i
                // 'moneda'       => $moneda, // si tu tabla de items lo tiene
            );
            $this->cotizacion_model->insert_item($item_data);
        }

        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            $this->session->set_flashdata('error', 'No se pudo actualizar la cotización. Inténtalo de nuevo.');
            redirect('cotizaciones/editar/' . $id);
            return;
        }

        $this->session->set_flashdata('success', 'Cotización actualizada exitosamente');
        redirect('cotizaciones/ver/' . $id);
    }

    public function eliminar($id) {
        $this->cotizacion_model->delete($id);
        $this->session->set_flashdata('success', 'Cotización eliminada exitosamente');
        redirect('cotizaciones');
    }

    /** ===== Exportar PDF ===== */
    public function exportar_pdf($id) {
        require_once APPPATH . 'libraries/tcpdf/tcpdf.php';
        
        $cotizacion = $this->cotizacion_model->get_by_id($id);
        $items      = $this->cotizacion_model->get_items($id);

        $moneda         = $cotizacion->moneda ?? 'MXN';
        $aplicar_iva    = (int)($cotizacion->aplicar_iva ?? 1);
        $iva_porcentaje = (float)($cotizacion->iva_porcentaje ?? 16);

        $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8');
        $pdf->AddFont('Poppins', '', 'poppins.php');
        $pdf->SetCreator('RaasMex');
        $pdf->SetAuthor('RaasMex');
        $pdf->SetTitle('Cotización ' . $cotizacion->numero);
    
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);
    
        $pdf->AddPage();
    
        // ===== Header imagen full =====
        $pdf->SetMargins(0, 0, 0);
        $pdf->SetAutoPageBreak(false, 0);
    
        $headerImg = FCPATH . 'assets/pdf/header_raasmex_2.png';
        $headerWmm = 210; // ancho A4
        $headerHmm = 50;  // alto del header
    
        if (is_file($headerImg)) {
            if ($info = @getimagesize($headerImg)) {
                [$pxW, $pxH] = $info;
                if ($pxW > 0) { $headerHmm = ($pxH / $pxW) * $headerWmm; }
            }
            $pdf->Image($headerImg, 0, 0, $headerWmm, $headerHmm, '', '', '', false, 300, '', false, false, 0);
        }
    
        // ===== Restablecer márgenes para contenido =====
        $offsetY = $headerHmm;
        $pdf->SetMargins(15, $offsetY, 15);
        $pdf->SetY($offsetY);
        $pdf->SetAutoPageBreak(true, 15);
        $pdf->SetFont('Poppins', '', 10);
    
        // PROPUESTA
        $htmlPropuesta = $this->html_propuesta($cotizacion);
        $pdf->writeHTML($htmlPropuesta, true, false, true, false, '');
    
        // Cintillo top
        $pageWidth   = $pdf->getPageWidth();
        $currentY    = $pdf->GetY() + 1;
        $cintilloPath= FCPATH . 'assets/pdf/top_divider.png';
        $pdf->Image($cintilloPath, 0, $currentY, $pageWidth, 4, '', '', '', false, 300, '', false, false, 0);
        $pdf->SetY($currentY + 6);
    
        // CLIENTE
        $htmlCliente = $this->html_cliente($cotizacion);
        $pdf->writeHTML($htmlCliente, true, false, true, false, '');

        // ITEMS (actualizado para usar moneda)
        $this->render_items_table_with_img_fullbleed($pdf, $items, $moneda);

        // TÉRMINOS + TOTALES (usa moneda e IVA desde BD)
        $this->render_terms_and_totals_section($pdf, $cotizacion, $moneda, $aplicar_iva, $iva_porcentaje);

        // Banda inferior
        $this->render_bottom_band($pdf);

        $pdf->Output('RAASMEX_Cotizacion_' . $cotizacion->numero . '.pdf', 'D');
    }

    private function html_propuesta($cotizacion) {
        $num = str_pad((string)$cotizacion->numero, 3, '0', STR_PAD_LEFT);
        ob_start(); ?>
        <div style="padding: 0 15mm; margin-top: 4mm;">
          <table style="width:100%; border-collapse:collapse;">
            <tr>
              <td style="text-align:center; vertical-align:middle; padding-top:2mm;">
                <div style="display:inline-block; text-align:center;">
                  <span style="font-size:20pt; color:#1e3150; font-weight:800; letter-spacing:1.5px; text-transform:uppercase;">
                    PROPUESTA
                  </span>
                  <span style="font-size:20pt; color:#1e3150; font-weight:800; margin-left:2mm;">
                    N°
                  </span>
                  <span style="font-size:20pt; color:#c0c6cd; font-weight:700; letter-spacing:2px; margin-left:2mm;">
                    <?php echo $num; ?>
                  </span>
                </div>
              </td>
            </tr>
          </table>
        </div>
        <?php
        return ob_get_clean();
    }
    
    private function html_cliente($cotizacion) {
        $fecha    = date('d/m/Y', strtotime((string)$cotizacion->fecha));
        $vigencia = (int)($cotizacion->vigencia_dias ?? 30);
        ob_start(); ?>
        <div style="margin-top:6mm;">
          <table style="width:100%; border-collapse:collapse; border-spacing:0;">
            <tr>
              <td style="width:75%; vertical-align:top; padding-right:10mm;">
                <table style="width:100%; border-collapse:collapse;">
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt; width:28mm;">Empresa:</td>
                    <td style="font-size:10pt; color:#111;"><?php echo htmlspecialchars($cotizacion->empresa); ?></td>
                  </tr>
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt;">Nombre:</td>
                    <td style="font-size:12pt; color:#111;"><?php echo htmlspecialchars($cotizacion->nombre); ?></td>
                  </tr>
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt;">Mail:</td>
                    <td style="font-size:10pt; color:#111;"><?php echo htmlspecialchars($cotizacion->email); ?></td>
                  </tr>
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt;">Dirección:</td>
                    <td style="font-size:10pt; color:#111;"><?php echo htmlspecialchars($cotizacion->direccion); ?></td>
                  </tr>
                </table>
              </td>
              <td style="width:25%; vertical-align:top;">
                <table style="width:100%; border-collapse:collapse;">
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt; width:22mm;">Fecha:</td>
                    <td style="font-size:9pt; color:#111;font-weight:bold;"><?php echo $fecha; ?></td>
                  </tr>
                  <tr>
                    <td style="color:#1e3150; font-weight:700; font-size:12pt;">Vigencia:</td>
                    <td style="font-size:10pt; color:#111;font-weight:bold;"><?php echo $vigencia; ?> días</td>
                  </tr>
                </table>
              </td>
            </tr>
          </table>
        </div>
        <?php
        return ob_get_clean();
    }

    private function render_items_table_with_img_fullbleed(TCPDF $pdf, array $items, $moneda = 'MXN') {
        $fmt = function($n) use ($moneda) { 
            return $this->fmt_money($n, $moneda);
        };
        // ---- Guardar estado actual
        $margins         = $pdf->getMargins();
        $savedAutoBreak  = $pdf->getAutoPageBreak();
        $savedBtmMargin  = $pdf->getBreakMargin();
    
        $pageW = $pdf->getPageWidth();
        $y     = $pdf->GetY();
        $h     = 8;
    
        // 1) Imagen header full-bleed
        $headerImg = FCPATH . 'assets/pdf/table_header_gradient.png';
        if (is_file($headerImg)) {
            $pdf->Image($headerImg, 0, $y, $pageW, $h, '', '', '', false, 300, '', false, false, 0);
        }
    
        // 2) Encabezados
        $htmlHeader = '
        <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
            <tr>
                <td style="width:8%; text-align:center; font-weight:900; font-size:12pt; color:#ffffff; letter-spacing:0.5px; padding:4mm 0 0 8mm;">
                    CANT.
                </td>
                <td style="width:22%; text-align:center; font-weight:900; font-size:12pt; color:#ffffff; letter-spacing:0.5px; padding:4mm 0 0 3mm;">
                    SERVICIO
                </td>
                <td style="width:30%; text-align:center; font-weight:900; font-size:12pt; color:#ffffff; letter-spacing:0.5px; padding:4mm 0 0 3mm;">
                    DESCRIPCIÓN
                </td>
                <td style="width:20%; text-align:center; font-weight:900; font-size:12pt; color:#ffffff; letter-spacing:0.5px; padding:4mm 8mm 0 0;">
                    P. UNITARIO
                </td>
                <td style="width:20%; text-align:center; font-weight:900; font-size:12pt; color:#ffffff; letter-spacing:0.5px; padding:4mm 8mm 0 0;">
                    TOTAL
                </td>
            </tr>
        </table>';
        $pdf->writeHTMLCell($pageW, $h, 0, $y, $htmlHeader, 0, 1, false, true, '', true);
    
        // 3) Posicionar cursor después del header
        $yBody = $y + $h;
        $pdf->SetY($yBody);
    
        // 4) Filas
        ob_start(); ?>
        <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
            <tbody>
            <?php foreach ($items as $it): ?>
                <tr style="vertical-align:top;">
                    <td style="width:8%; text-align:center; font-weight:800; color:#1e3150; font-size:11pt; padding:3mm 0 3mm 8mm;">
                        <?php echo (int)$it->cantidad; ?>
                    </td>
                    
                    <td style="width:22%; text-align:center; font-weight:700; color:#2c4a6d; font-size:9.5pt; line-height:1.4; padding:3mm 0 3mm 3mm;">
                        <?php echo nl2br(htmlspecialchars($it->servicio)); ?>
                    </td>
                    
                    <td style="width:30%; text-align:left; font-size:7pt; line-height:1.5; color:#333333; padding:3mm 3mm 3mm 3mm;">
                        <?php 
                        $descripcion = $it->descripcion;
                        $descripcion = str_replace(['<script', '</script'], ['&lt;script', '&lt;/script'], $descripcion);
                        echo nl2br($descripcion); 
                        ?>
                    </td>
                    
                    <td style="width:20%; text-align:right; font-size:11pt; font-weight:700; color:#5a6c7d; padding:3mm 8mm 3mm 0;">
                        <?php echo $fmt($it->precio_unitario); ?>
                    </td>
                    
                    <td style="width:20%; text-align:right; font-size:11pt; font-weight:800; color:#5a6c7d; padding:3mm 8mm 3mm 0;">
                        <?php echo $fmt($it->total); ?>
                    </td>
                </tr>
                <tr>
                    <td colspan="5" style="border-bottom:1pt solid #e3eaf1; padding:0; height:0;"></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        <?php
        $htmlBody = ob_get_clean();
    
        // full-bleed
        $pdf->SetAutoPageBreak(false, 0);
        $pdf->SetMargins(0, $yBody, 0);
        $pdf->writeHTMLCell($pageW, 0, 0, $yBody, $htmlBody, 0, 1, false, true, '', true);
    
        // Restaurar
        $newY = $pdf->GetY() + 5;
        $pdf->SetMargins($margins['left'], $margins['top'], $margins['right']);
        $pdf->SetY($newY);
        $pdf->SetAutoPageBreak($savedAutoBreak, $savedBtmMargin);
    }

    private function render_terms_and_totals_section(TCPDF $pdf, $cotizacion, $moneda = 'MXN', $aplicar_iva = 1, $iva_porcentaje = 16) {
        $startY = $pdf->GetY() + 2;
        $pageW  = $pdf->getPageWidth();
    
        // ===== Columna izquierda: TÉRMINOS =====
        $xStart = 8;
        $y = $startY;
    
        $pdf->SetFont('Poppins', 'B', 13);
        $pdf->SetTextColor(44, 74, 109);
        $pdf->SetXY($xStart, $y);
        $pdf->Cell(0, 0, 'TÉRMINOS Y CONDICIONES', 0, 1, 'L');
        $y = $pdf->GetY() + 3;
    
        $observaciones = isset($cotizacion->observaciones) && !empty($cotizacion->observaciones) 
            ? $cotizacion->observaciones 
            : "Se requiere el pago de la implementación al confirmar el servicio.
Los precios no incluyen I.V.A.
Vigencia del Servicio: 12 meses. Cancelación sin penalización con aviso anticipado de 25 días.
Facturación de la licencia: Se enviará a inicios de cada mes con un plazo de crédito de 25 días.";
    
        $lines = explode("\n", trim($observaciones));
        $pdf->SetFont('Poppins', '', 8);
        $pdf->SetTextColor(51, 51, 51);
        $termsMaxWidth = ($pageW * 0.50);
    
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;
            $line = preg_replace('/^[\-\*]\s*/', '', $line);
            $pdf->SetXY($xStart, $y);
            $pdf->SetFont('Poppins', 'B', 8);
            $pdf->SetTextColor(44, 74, 109);
            $pdf->Write(0, '- ', '', false, 'L', false);
            $xAfterDash = $pdf->GetX();
            $yLine = $pdf->GetY();
            $pdf->SetFont('Poppins', '', 8);
            $pdf->SetTextColor(51, 51, 51);
            $pdf->SetXY($xAfterDash, $yLine);
            $availableWidth = $termsMaxWidth - $xAfterDash;
            $pdf->MultiCell($availableWidth, 0, $line, 0, 'L', false, 1, '', '', true, 0, false, true, 0, 'T', false);
            $y = $pdf->GetY() + 1;
        }
    
        // ===== Columna derecha: TOTALES =====
        $fmt = function($n) use ($moneda) { return $this->fmt_money($n, $moneda); };

        $subtotal  = (float)($cotizacion->subtotal ?? 0);
        $descuento = (float)($cotizacion->descuento ?? 0);
        if ($descuento < 0) $descuento = 0;
        if ($descuento > $subtotal) $descuento = $subtotal;
        $base  = $subtotal - $descuento;

        // Usa valores de BD si existen
        $iva   = isset($cotizacion->iva)   ? (float)$cotizacion->iva   : round($base * ($iva_porcentaje/100), 2);
        $total = isset($cotizacion->total) ? (float)$cotizacion->total : round($base + $iva, 2);

        $title = isset($cotizacion->tipo_inversion) ? $cotizacion->tipo_inversion : 'INVERSIÓN MENSUAL';
        $blockWidth   = $pageW * 0.40;
        $xStartTotals = $pageW - $blockWidth;
        $y = $startY;

        $pdf->AddFont('Poppins', 'SB', 'poppinssb.php'); // SemiBold
        $pdf->SetFont('Poppins', 'SB', 13);
        $pdf->SetTextColor(91, 164, 207);
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, strtoupper($title), 0, 1, false, true, 'R', true);
        $y = $pdf->GetY() + 2;

        // SUBTOTAL
        $htmlSubtotal = '
        <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
          <tr>
            <td style="width:50%; text-align:left; font-size:11pt; font-weight:800; color:#1e3150; padding:2mm 0 2mm 0;">
              SUBTOTAL:
            </td>
            <td style="width:50%; text-align:right; font-size:11pt; font-weight:400; color:#7a8a9a; padding:2mm 8mm 2mm 0;">
              '.$fmt($subtotal).'
            </td>
          </tr>
        </table>';
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlSubtotal, 0, 1, false, true, '', true);
        $y = $pdf->GetY();

        // DESCUENTO (si aplica)
        if ($descuento > 0) {
            $htmlDescuento = '
            <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
              <tr>
                <td style="width:50%; text-align:left; font-size:11pt; font-weight:800; color:#1e3150; padding:2mm 0 2mm 0;">
                  DESCUENTO:
                </td>
                <td style="width:50%; text-align:right; font-size:11pt; font-weight:400; color:#c0392b; padding:2mm 8mm 2mm 0;">
                  - '.$fmt($descuento).'
                </td>
              </tr>
            </table>';
            $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlDescuento, 0, 1, false, true, '', true);
            $y = $pdf->GetY();
        }

        // IVA (porcentaje dinámico)
        $ivaLabelPct = $aplicar_iva ? (strpos((string)$iva_porcentaje, '.') === false ? (int)$iva_porcentaje : $iva_porcentaje) : 0;
        $htmlIva = '
        <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
          <tr>
            <td style="width:50%; text-align:left; font-size:11pt; font-weight:800; color:#1e3150; padding:2mm 0 2mm 0;">
              IVA. '.$ivaLabelPct.'%:
            </td>
            <td style="width:50%; text-align:right; font-size:11pt; font-weight:400; color:#7a8a9a; padding:2mm 8mm 2mm 0;">
              '.$fmt($iva).'
            </td>
          </tr>
        </table>';
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlIva, 0, 1, false, true, '', true);
        $y = $pdf->GetY() + 1;

        // Línea y TOTAL
        $pdf->Line($xStartTotals, $y, $xStartTotals + $blockWidth, $y, ['width' => 0.5, 'color' => [30, 49, 80]]);
        $y += 3;

        $htmlTotal = '
        <table cellpadding="0" cellspacing="0" border="0" style="width:100%;">
          <tr>
            <td style="width:40%; text-align:left; font-size:12pt; font-weight:900; color:#1e3150; padding:2mm 0 2mm 0;">
              TOTAL:
            </td>
            <td style="width:60%; text-align:right; font-size:12pt; font-weight:bold; color:#7a8a9a; padding:2mm 8mm 2mm 0;">
              '.$fmt($total).'
            </td>
          </tr>
        </table>';
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlTotal, 0, 1, false, true, '', true);

        // Firma (igual que tenías)
        $y = $pdf->GetY() + 10;
        $signatureImg = FCPATH . 'assets/pdf/signature.png';
        if (is_file($signatureImg)) {
            $imgWidth = 40; $imgHeight = 20;
            $xImg = $xStartTotals + (($blockWidth - $imgWidth - 8) / 2);
            $pdf->Image($signatureImg, $xImg, $y, $imgWidth, $imgHeight, '', '', '', false, 300, '', false, false, 0);
            $y += $imgHeight + 3;
        } else {
            $y += 25;
        }

        $lineWidth = 60;
        $xLine = $xStartTotals + $blockWidth - $lineWidth - 8;
        $pdf->Line($xLine, $y, $xLine + $lineWidth, $y, ['width' => 0.5, 'color' => [44, 74, 109]]);
        $y += 1;

        $htmlNombre = '
        <div style="text-align:center; font-size:11pt; font-weight:900; color:#2c4a6d; padding-right:8mm;">
          '.(isset($cotizacion->firma_nombre) ? strtoupper($cotizacion->firma_nombre) : 'JESÚS RODRÍGUEZ MORALES').'
        </div>';
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlNombre, 0, 1, false, true, 'C', true);
        $y = $pdf->GetY() + 1;

        $htmlCargo = '
        <div style="text-align:center; font-size:10pt; font-weight:400; color:#646464; padding-right:8mm;">
          '.(isset($cotizacion->firma_cargo) ? $cotizacion->firma_cargo : 'FOUNDER - CEO').'
        </div>';
        $pdf->writeHTMLCell($blockWidth, 0, $xStartTotals, $y, $htmlCargo, 0, 1, false, true, 'C', true);
    }

    private function render_bottom_band(TCPDF $pdf) {
        $margins  = $pdf->getMargins();
        $pageW    = $pdf->getPageWidth();
        $pageH    = $pdf->getPageHeight();
        $bandWidth  = $pageW - $margins['left'] - $margins['right'];
        $xStart     = $margins['left'];
        $bandHeight = 6;
        $bottomMargin = 15;
        $currentY = $pageH - $bottomMargin - $bandHeight;
        $cintilloPath = FCPATH . 'assets/pdf/bottom_pill.png';
        if (is_file($cintilloPath)) {
            $pdf->Image($cintilloPath, $xStart, $currentY, $bandWidth, $bandHeight, '', '', '', false, 300, '', false, false, 0);
        }
    }
}
