<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Editar Cotización <?= htmlspecialchars($cotizacion->numero) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php
        // ===== Valores por defecto desde la BD =====
        $moneda         = isset($cotizacion->moneda) ? $cotizacion->moneda : 'MXN';
        $aplicar_iva    = isset($cotizacion->aplicar_iva) ? (int)$cotizacion->aplicar_iva : 1;
        $iva_porcentaje = isset($cotizacion->iva_porcentaje) ? (float)$cotizacion->iva_porcentaje : 16;

        // Helpers de moneda
        $currency_symbol = function($code) {
            $c = strtoupper((string)$code);
            return $c === 'USD' ? '$' : '$'; // cambia a 'US$' si prefieres
        };
        $currency_suffix = function($code) {
            $c = strtoupper((string)$code);
            return in_array($c, ['MXN','USD']) ? $c : 'MXN';
        };
        $fmt = function($n, $code) use ($currency_symbol, $currency_suffix) {
            return $currency_symbol($code) . number_format((float)$n, 2, '.', ',') . ' ' . $currency_suffix($code);
        };
    ?>

    <div class="container mt-5">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2><i class="fas fa-edit"></i> Editar Cotización <?= htmlspecialchars($cotizacion->numero) ?></h2>
            <a href="<?= base_url('index.php/cotizaciones/ver/' . $cotizacion->id) ?>" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Cancelar
            </a>
        </div>

        <form action="<?= base_url('index.php/cotizaciones/actualizar/' . $cotizacion->id) ?>" method="POST">
            <input type="hidden" name="<?= $this->security->get_csrf_token_name(); ?>" value="<?= $this->security->get_csrf_hash(); ?>">

            <div class="card mb-3">
                <div class="card-header">
                    <h5>Información General</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Número de Cotización</label>
                            <input type="text" class="form-control" value="<?= htmlspecialchars($cotizacion->numero) ?>" readonly>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Fecha *</label>
                            <input type="date" name="fecha" class="form-control" value="<?= htmlspecialchars($cotizacion->fecha) ?>" required>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Vigencia (días) *</label>
                            <input type="number" name="vigencia_dias" class="form-control" value="<?= (int)$cotizacion->vigencia_dias ?>" required>
                        </div>

                        <!-- Moneda -->
                        <div class="col-md-3">
                            <label class="form-label">Moneda *</label>
                            <select name="moneda" id="moneda" class="form-select" required>
                                <option value="MXN" <?= $moneda === 'MXN' ? 'selected' : '' ?>>MXN – Peso mexicano</option>
                                <option value="USD" <?= $moneda === 'USD' ? 'selected' : '' ?>>USD – Dólar estadounidense</option>
                            </select>
                        </div>

                        <!-- Tipo de inversión -->
                        <div class="col-md-12">
                            <div class="mb-1">
                                <label class="form-label d-block">Tipo de Inversión *</label>
                                <?php
                                    $tipoSel = isset($cotizacion->tipo_inversion) && $cotizacion->tipo_inversion !== ''
                                        ? $cotizacion->tipo_inversion
                                        : 'INVERSIÓN MENSUAL';
                                ?>
                                <div class="form-check form-check-inline">
                                    <input class="form-check-input" type="radio" name="tipo_inversion" id="inversion_mensual" value="INVERSIÓN MENSUAL" <?= $tipoSel === 'INVERSIÓN MENSUAL' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="inversion_mensual">INVERSIÓN MENSUAL</label>
                                </div>
                                <div class="form-check form-check-inline">
                                    <input class="form-check-input" type="radio" name="tipo_inversion" id="inversion_implementacion" value="INVERSIÓN IMPLEMENTACIÓN" <?= $tipoSel === 'INVERSIÓN IMPLEMENTACIÓN' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="inversion_implementacion">INVERSIÓN IMPLEMENTACIÓN</label>
                                </div>
                                <div class="form-check form-check-inline">
                                    <input class="form-check-input" type="radio" name="tipo_inversion" id="inversion_inicial" value="INVERSIÓN INICIAL CON 1 MES DE LICENCIA" <?= $tipoSel === 'INVERSIÓN INICIAL CON 1 MES DE LICENCIA' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="inversion_inicial">INVERSIÓN INICIAL CON 1 MES DE LICENCIA</label>
                                </div>
                                <div class="form-check form-check-inline">
                                    <input class="form-check-input" type="radio" name="tipo_inversion" id="inversion_total_anual" value="INVERSIÓN TOTAL ANUAL" <?= $tipoSel === 'INVERSIÓN TOTAL ANUAL' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="inversion_total_anual">INVERSIÓN TOTAL ANUAL</label>
                                </div>
                            </div>
                        </div>

                        <!-- Cliente -->
                        <div class="col-md-12">
                            <label class="form-label">Cliente *</label>
                            <select name="cliente_id" class="form-select" required>
                                <option value="">Seleccione un cliente</option>
                                <?php foreach($clientes as $cliente): ?>
                                    <option value="<?= $cliente->id ?>" <?= $cliente->id == $cotizacion->cliente_id ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($cliente->nombre) ?> <?= $cliente->empresa ? '- ' . htmlspecialchars($cliente->empresa) : '' ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                    </div>
                </div>
            </div>

            <!-- Items -->
            <div class="card mb-3">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5>Items de la Cotización</h5>
                    <button type="button" class="btn btn-sm btn-primary" onclick="agregarItem()">
                        <i class="fas fa-plus"></i> Agregar Item
                    </button>
                </div>
                <div class="card-body">
                    <div id="items-container">
                        <?php foreach($items as $item): ?>
                            <div class="item-row border p-3 mb-3 rounded">
                                <div class="row g-3">
                                    <div class="col-md-3">
                                        <label class="form-label">Servicio *</label>
                                        <input type="text" name="servicio[]" class="form-control" value="<?= htmlspecialchars($item->servicio) ?>" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Descripción *</label>
                                        <textarea name="descripcion[]" class="form-control" rows="3" required><?= htmlspecialchars($item->descripcion) ?></textarea>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label">Cantidad *</label>
                                        <input type="number" name="cantidad[]" class="form-control cantidad" value="<?= (float)$item->cantidad ?>" min="1" required oninput="calcularTotales()">
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label">Precio Unit. *</label>
                                        <input type="number" step="any" name="precio_unitario[]" class="form-control precio" value="<?= (float)$item->precio_unitario ?>" required oninput="calcularTotales()">
                                    </div>
                                    <div class="col-md-1 d-flex align-items-end">
                                        <button type="button" class="btn btn-danger btn-sm w-100" onclick="eliminarItem(this)" style="margin-bottom: 8px;">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- IVA controls -->
                    <div class="row mt-3 g-3">
                        <div class="col-md-3">
                            <div class="form-check mt-2">
                                <input class="form-check-input" type="checkbox" id="aplicar_iva" name="aplicar_iva" <?= $aplicar_iva ? 'checked' : '' ?>>
                                <label class="form-check-label" for="aplicar_iva">Aplicar IVA</label>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">IVA (%)</label>
                            <input type="number" step="any" min="0" id="iva_porcentaje" name="iva_porcentaje" class="form-control" value="<?= htmlspecialchars($iva_porcentaje) ?>" oninput="calcularTotales()">
                            <div class="form-text">MXN típico: 16% • USD típico: 0%</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Observaciones -->
            <div class="card mb-3">
                <div class="card-header">
                    <h5>Observaciones</h5>
                </div>
                <div class="card-body">
                    <textarea name="observaciones" class="form-control" rows="5" placeholder="Términos y condiciones adicionales..."><?= htmlspecialchars($cotizacion->observaciones) ?></textarea>
                </div>
            </div>

            <!-- Totales -->
            <div class="card mb-3">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8"></div>
                        <div class="col-md-4">
                            <table class="table">
                                <tr>
                                    <td><strong>Subtotal:</strong></td>
                                    <td class="text-end">
                                        <span id="currencySymbolSub"><?= $currency_symbol($moneda) ?></span>
                                        <span id="subtotal">0.00</span>
                                        <span id="currencyCodeSub"><?= $currency_suffix($moneda) ?></span>
                                    </td>
                                </tr>
                                <tr>
                                    <td class="align-middle"><strong>Descuento:</strong></td>
                                    <td class="text-end">
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text" id="currencySymbolDesc"><?= $currency_symbol($moneda) ?></span>
                                            <input
                                                type="number"
                                                name="descuento"
                                                id="descuento"
                                                class="form-control text-end"
                                                step="any"
                                                min="0"
                                                value="<?= number_format((float)($cotizacion->descuento ?? 0), 2, '.', '') ?>"
                                                oninput="calcularTotales()">
                                            <span class="input-group-text" id="currencyCodeDesc"><?= $currency_suffix($moneda) ?></span>
                                        </div>
                                        <small class="text-muted d-block">El descuento se aplica antes del IVA.</small>
                                    </td>
                                </tr>
                                <tr>
                                    <td><strong>Base (Subt. − Desc.):</strong></td>
                                    <td class="text-end">
                                        <span id="currencySymbolBase"><?= $currency_symbol($moneda) ?></span>
                                        <span id="base">0.00</span>
                                        <span id="currencyCodeBase"><?= $currency_suffix($moneda) ?></span>
                                    </td>
                                </tr>
                                <tr>
                                    <td>
                                        <strong>IVA (<span id="ivaLabel">0</span>%):</strong>
                                    </td>
                                    <td class="text-end">
                                        <span id="currencySymbolIva"><?= $currency_symbol($moneda) ?></span>
                                        <span id="iva">0.00</span>
                                        <span id="currencyCodeIva"><?= $currency_suffix($moneda) ?></span>
                                    </td>
                                </tr>
                                <tr class="table-primary">
                                    <td><strong>TOTAL:</strong></td>
                                    <td class="text-end">
                                        <strong>
                                            <span id="currencySymbolTotal"><?= $currency_symbol($moneda) ?></span>
                                            <span id="total">0.00</span>
                                            <span id="currencyCodeTotal"><?= $currency_suffix($moneda) ?></span>
                                        </strong>
                                    </td>
                                </tr>
                            </table>

                            <div class="text-end">
                                <span class="badge bg-secondary">Importes en <span id="badgeCurrency"><?= $currency_suffix($moneda) ?></span></span>
                                <span class="badge bg-light text-dark border ms-1" id="badgeIva" style="<?= $aplicar_iva ? 'display:none;' : '' ?>">IVA no aplicado</span>
                            </div>

                        </div>
                    </div>
                </div>
            </div>

            <div class="text-end mb-5">
                <button type="submit" class="btn btn-success btn-lg">
                    <i class="fas fa-save"></i> Actualizar Cotización
                </button>
            </div>
        </form>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // ===== Helpers =====
        function toNumber(v) {
            const n = parseFloat(String(v).replace(/,/g, ''));
            return isFinite(n) ? n : 0;
        }
        function format2(n) {
            return (isFinite(n) ? n : 0).toFixed(2);
        }

        // ===== Referencias UI =====
        const monedaSelect   = document.getElementById('moneda');
        const aplicarIvaChk  = document.getElementById('aplicar_iva');
        const ivaPctInput    = document.getElementById('iva_porcentaje');
        const ivaLabel       = document.getElementById('ivaLabel');
        const badgeCurrency  = document.getElementById('badgeCurrency');
        const badgeIva       = document.getElementById('badgeIva');

        // Símbolo/Sufijo
        function getCurrencyMeta(code) {
            code = (code || 'MXN').toUpperCase();
            if (code === 'USD') return { symbol: '$', code: 'USD', defaultIva: 0 };
            return { symbol: '$', code: 'MXN', defaultIva: 16 };
        }

        function syncCurrencyUi() {
            const meta = getCurrencyMeta(monedaSelect.value);

            // Top badges
            badgeCurrency.textContent = meta.code;
            badgeIva.style.display = aplicarIvaChk.checked ? 'none' : 'inline-block';

            // Update symbols/codes in totals table
            document.getElementById('currencySymbolSub').textContent   = meta.symbol;
            document.getElementById('currencySymbolDesc').textContent  = meta.symbol;
            document.getElementById('currencySymbolBase').textContent  = meta.symbol;
            document.getElementById('currencySymbolIva').textContent   = meta.symbol;
            document.getElementById('currencySymbolTotal').textContent = meta.symbol;

            document.getElementById('currencyCodeSub').textContent   = meta.code;
            document.getElementById('currencyCodeDesc').textContent  = meta.code;
            document.getElementById('currencyCodeBase').textContent  = meta.code;
            document.getElementById('currencyCodeIva').textContent   = meta.code;
            document.getElementById('currencyCodeTotal').textContent = meta.code;

            // Ajustar IVA típico al cambiar moneda solo si el usuario no lo ha modificado (heurística simple)
            if (meta.code === 'USD') {
                if (parseFloat(ivaPctInput.value) === 16) ivaPctInput.value = meta.defaultIva;
                aplicarIvaChk.checked = (parseFloat(ivaPctInput.value) > 0);
            } else if (meta.code === 'MXN') {
                if (parseFloat(ivaPctInput.value) === 0) ivaPctInput.value = meta.defaultIva;
                aplicarIvaChk.checked = true;
            }

            ivaPctInput.disabled = !aplicarIvaChk.checked;
            ivaLabel.textContent = aplicarIvaChk.checked ? ((+ivaPctInput.value % 1 === 0) ? parseInt(ivaPctInput.value) : parseFloat(ivaPctInput.value).toFixed(2)) : '0';

            calcularTotales();
        }

        // ===== Items / Totales =====
        function agregarItem() {
            const container = document.getElementById('items-container');
            const newItem = document.createElement('div');
            newItem.className = 'item-row border p-3 mb-3 rounded';
            newItem.innerHTML = `
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Servicio *</label>
                        <input type="text" name="servicio[]" class="form-control" required>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Descripción *</label>
                        <textarea name="descripcion[]" class="form-control" rows="3" required></textarea>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Cantidad *</label>
                        <input type="number" name="cantidad[]" class="form-control cantidad" value="1" min="1" required oninput="calcularTotales()">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Precio Unit. *</label>
                        <input type="number" step="any" name="precio_unitario[]" class="form-control precio" required oninput="calcularTotales()">
                    </div>
                    <div class="col-md-1 d-flex align-items-end">
                        <button type="button" class="btn btn-danger btn-sm w-100" onclick="eliminarItem(this)" style="margin-bottom: 8px;">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
            `;
            container.appendChild(newItem);
        }

        function eliminarItem(btn) {
            const items = document.querySelectorAll('.item-row');
            if (items.length > 1) {
                btn.closest('.item-row').remove();
                calcularTotales();
            } else {
                alert('Debe haber al menos un item en la cotización');
            }
        }

        function calcularTotales() {
            const cantidades = document.querySelectorAll('.cantidad');
            const precios    = document.querySelectorAll('.precio');
            let subtotal = 0;

            for (let i = 0; i < cantidades.length; i++) {
                const cantidad = toNumber(cantidades[i].value);
                const precio   = toNumber(precios[i].value);
                subtotal += cantidad * precio;
            }

            // Descuento
            const descInput = document.getElementById('descuento');
            let descuento = toNumber(descInput.value);
            if (descuento < 0) descuento = 0;
            if (descuento > subtotal) descuento = subtotal;
            descInput.value = format2(descuento);

            const base = subtotal - descuento;

            // IVA
            let iva = 0;
            if (aplicarIvaChk.checked) {
                const pct = Math.max(0, toNumber(ivaPctInput.value));
                iva = base * (pct / 100);
                ivaLabel.textContent = (pct % 1 === 0) ? pct.toString() : pct.toFixed(2);
            } else {
                iva = 0;
                ivaLabel.textContent = '0';
            }

            const total = base + iva;

            // Pintar
            document.getElementById('subtotal').textContent = format2(subtotal);
            document.getElementById('base').textContent     = format2(base);
            document.getElementById('iva').textContent      = format2(iva);
            document.getElementById('total').textContent    = format2(total);

            // Estado badges
            badgeIva.style.display = aplicarIvaChk.checked ? 'none' : 'inline-block';
        }

        // ===== Eventos =====
        document.addEventListener('DOMContentLoaded', () => {
            // Inicializa UI con los valores del servidor:
            // Pre-cálculo usando valores actuales del DOM (items + descuento + iva settings)
            syncCurrencyUi();
            calcularTotales();
        });

        monedaSelect.addEventListener('change', syncCurrencyUi);
        aplicarIvaChk.addEventListener('change', () => {
            ivaPctInput.disabled = !aplicarIvaChk.checked;
            calcularTotales();
        });
        ivaPctInput.addEventListener('input', calcularTotales);
    </script>
</body>
</html>
